/**
 * FICHIER JAVASCRIPT POUR L'ACCESSIBILITÉ ET L'INTERACTIVITÉ
 * ===========================================================
 */

(function() {
    'use strict';

    // Attendre que le DOM soit chargé
    document.addEventListener('DOMContentLoaded', function() {
        
        // Initialiser tous les composants d'accessibilité
        initAccessibility();
        
        // Initialiser le menu mobile
        initMobileMenu();
        
        // Initialiser le bouton "retour en haut"
        initBackToTop();
        
        // Initialiser la gestion du focus
        initFocusManagement();
        
        // Initialiser les formulaires
        initForms();
        
        // Initialiser les animations (si l'utilisateur n'a pas demandé de réduire les animations)
        if (!window.matchMedia('(prefers-reduced-motion: reduce)').matches) {
            initAnimations();
        }
    });

    /**
     * Initialisation générale de l'accessibilité
     */
    function initAccessibility() {
        // Ajouter des attributs ARIA manquants
        addMissingAriaAttributes();
        
        // Améliorer la navigation au clavier
        improveKeyboardNavigation();
        
        // Ajouter des raccourcis clavier
        addKeyboardShortcuts();
    }

    /**
     * Ajouter des attributs ARIA manquants
     */
    function addMissingAriaAttributes() {
        // Ajouter aria-label aux images sans alt
        const images = document.querySelectorAll('img:not([alt])');
        images.forEach(function(img) {
            if (!img.alt) {
                img.setAttribute('alt', 'Image');
            }
        });

        // Ajouter role aux listes
        const lists = document.querySelectorAll('ul, ol');
        lists.forEach(function(list) {
            if (!list.getAttribute('role')) {
                list.setAttribute('role', 'list');
            }
        });

        // Ajouter role aux éléments de liste
        const listItems = document.querySelectorAll('li');
        listItems.forEach(function(item) {
            if (!item.getAttribute('role')) {
                item.setAttribute('role', 'listitem');
            }
        });
    }

    /**
     * Améliorer la navigation au clavier
     */
    function improveKeyboardNavigation() {
        // Gérer la navigation dans les menus
        const menuItems = document.querySelectorAll('.nav-menu a, .mobile-nav-menu a');
        
        menuItems.forEach(function(item, index) {
            item.addEventListener('keydown', function(e) {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    this.click();
                }
            });
        });

        // Gérer la navigation avec Tab
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Tab') {
                document.body.classList.add('keyboard-navigation');
            }
        });

        document.addEventListener('mousedown', function() {
            document.body.classList.remove('keyboard-navigation');
        });
    }

    /**
     * Ajouter des raccourcis clavier
     */
    function addKeyboardShortcuts() {
        document.addEventListener('keydown', function(e) {
            // Alt + H : Aller à l'accueil
            if (e.altKey && e.key === 'h') {
                e.preventDefault();
                window.location.href = '/';
            }
            
            // Alt + S : Aller aux services
            if (e.altKey && e.key === 's') {
                e.preventDefault();
                window.location.href = '/our-services/';
            }
            
            // Alt + C : Aller au contact
            if (e.altKey && e.key === 'c') {
                e.preventDefault();
                window.location.href = '/contact-us/';
            }
            
            // Échap : Fermer le menu mobile
            if (e.key === 'Escape') {
                const mobileMenu = document.getElementById('mobile-menu');
                const mobileMenuButton = document.querySelector('.mobile-menu-icon');
                
                if (mobileMenu && mobileMenu.classList.contains('active')) {
                    closeMobileMenu();
                    mobileMenuButton.focus();
                }
            }
        });
    }

    /**
     * Initialiser le menu mobile
     */
    function initMobileMenu() {
        const mobileMenuButton = document.querySelector('.mobile-menu-icon');
        const mobileMenu = document.getElementById('mobile-menu');
        
        if (!mobileMenuButton || !mobileMenu) return;

        mobileMenuButton.addEventListener('click', function() {
            const isExpanded = this.getAttribute('aria-expanded') === 'true';
            
            if (isExpanded) {
                closeMobileMenu();
            } else {
                openMobileMenu();
            }
        });

        // Fermer le menu en cliquant sur un lien
        const mobileMenuLinks = mobileMenu.querySelectorAll('a');
        mobileMenuLinks.forEach(function(link) {
            link.addEventListener('click', function() {
                closeMobileMenu();
            });
        });

        // Fermer le menu en cliquant à l'extérieur
        document.addEventListener('click', function(e) {
            if (!mobileMenu.contains(e.target) && !mobileMenuButton.contains(e.target)) {
                closeMobileMenu();
            }
        });
    }

    /**
     * Ouvrir le menu mobile
     */
    function openMobileMenu() {
        const mobileMenuButton = document.querySelector('.mobile-menu-icon');
        const mobileMenu = document.getElementById('mobile-menu');
        
        mobileMenuButton.setAttribute('aria-expanded', 'true');
        mobileMenu.classList.add('active');
        mobileMenu.setAttribute('aria-hidden', 'false');
        
        // Focus sur le premier lien du menu
        const firstLink = mobileMenu.querySelector('a');
        if (firstLink) {
            firstLink.focus();
        }
        
        // Empêcher le défilement de la page
        document.body.style.overflow = 'hidden';
    }

    /**
     * Fermer le menu mobile
     */
    function closeMobileMenu() {
        const mobileMenuButton = document.querySelector('.mobile-menu-icon');
        const mobileMenu = document.getElementById('mobile-menu');
        
        mobileMenuButton.setAttribute('aria-expanded', 'false');
        mobileMenu.classList.remove('active');
        mobileMenu.setAttribute('aria-hidden', 'true');
        
        // Restaurer le défilement de la page
        document.body.style.overflow = '';
    }

    /**
     * Initialiser le bouton "retour en haut"
     */
    function initBackToTop() {
        const backToTopButton = document.getElementById('back-to-top');
        
        if (!backToTopButton) return;

        // Afficher/masquer le bouton selon le défilement
        window.addEventListener('scroll', function() {
            if (window.pageYOffset > 300) {
                backToTopButton.classList.add('visible');
            } else {
                backToTopButton.classList.remove('visible');
            }
        });

        // Gérer le clic sur le bouton
        backToTopButton.addEventListener('click', function() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        });

        // Gérer la navigation au clavier
        backToTopButton.addEventListener('keydown', function(e) {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                this.click();
            }
        });
    }

    /**
     * Initialiser la gestion du focus
     */
    function initFocusManagement() {
        // Gérer le focus visible
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Tab') {
                document.body.classList.add('focus-visible');
            }
        });

        document.addEventListener('mousedown', function() {
            document.body.classList.remove('focus-visible');
        });

        // Gérer le focus dans les modales et menus
        const focusableElements = 'button, [href], input, select, textarea, [tabindex]:not([tabindex="-1"])';
        
        function trapFocus(element) {
            const focusableContent = element.querySelectorAll(focusableElements);
            const firstFocusableElement = focusableContent[0];
            const lastFocusableElement = focusableContent[focusableContent.length - 1];

            document.addEventListener('keydown', function(e) {
                if (e.key === 'Tab') {
                    if (e.shiftKey) {
                        if (document.activeElement === firstFocusableElement) {
                            lastFocusableElement.focus();
                            e.preventDefault();
                        }
                    } else {
                        if (document.activeElement === lastFocusableElement) {
                            firstFocusableElement.focus();
                            e.preventDefault();
                        }
                    }
                }
            });
        }

        // Appliquer le piège de focus au menu mobile
        const mobileMenu = document.getElementById('mobile-menu');
        if (mobileMenu) {
            trapFocus(mobileMenu);
        }
    }

    /**
     * Initialiser les formulaires
     */
    function initForms() {
        const forms = document.querySelectorAll('form');
        
        forms.forEach(function(form) {
            // Gérer la validation
            form.addEventListener('submit', function(e) {
                if (!validateForm(this)) {
                    e.preventDefault();
                }
            });

            // Gérer les états de chargement
            form.addEventListener('submit', function() {
                const submitButton = this.querySelector('button[type="submit"]');
                if (submitButton) {
                    submitButton.classList.add('loading');
                    submitButton.disabled = true;
                }
            });
        });
    }

    /**
     * Valider un formulaire
     */
    function validateForm(form) {
        let isValid = true;
        const inputs = form.querySelectorAll('input[required], textarea[required], select[required]');
        
        inputs.forEach(function(input) {
            const formGroup = input.closest('.form-group');
            
            // Supprimer les erreurs précédentes
            const existingError = formGroup.querySelector('.error-message');
            if (existingError) {
                existingError.remove();
            }
            formGroup.classList.remove('has-error');
            
            // Valider le champ
            if (!input.value.trim()) {
                isValid = false;
                formGroup.classList.add('has-error');
                
                const errorMessage = document.createElement('div');
                errorMessage.className = 'error-message';
                errorMessage.textContent = 'Ce champ est requis.';
                formGroup.appendChild(errorMessage);
            }
            
            // Validation spécifique pour les emails
            if (input.type === 'email' && input.value.trim()) {
                const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                if (!emailRegex.test(input.value)) {
                    isValid = false;
                    formGroup.classList.add('has-error');
                    
                    const errorMessage = document.createElement('div');
                    errorMessage.className = 'error-message';
                    errorMessage.textContent = 'Veuillez entrer une adresse email valide.';
                    formGroup.appendChild(errorMessage);
                }
            }
        });
        
        return isValid;
    }

    /**
     * Initialiser les animations
     */
    function initAnimations() {
        // Animation d'apparition au défilement
        const observerOptions = {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        };

        const observer = new IntersectionObserver(function(entries) {
            entries.forEach(function(entry) {
                if (entry.isIntersecting) {
                    entry.target.classList.add('animate-in');
                }
            });
        }, observerOptions);

        // Observer les éléments avec la classe wow
        const animatedElements = document.querySelectorAll('.wow');
        animatedElements.forEach(function(element) {
            observer.observe(element);
        });
    }

    /**
     * Utilitaires pour l'accessibilité
     */
    window.AccessibilityUtils = {
        // Annoncer un message aux lecteurs d'écran
        announceToScreenReader: function(message) {
            const announcement = document.createElement('div');
            announcement.setAttribute('aria-live', 'polite');
            announcement.setAttribute('aria-atomic', 'true');
            announcement.className = 'sr-only';
            announcement.textContent = message;
            
            document.body.appendChild(announcement);
            
            setTimeout(function() {
                document.body.removeChild(announcement);
            }, 1000);
        },

        // Vérifier si un élément est visible
        isElementVisible: function(element) {
            const rect = element.getBoundingClientRect();
            return rect.top >= 0 && rect.left >= 0 && 
                   rect.bottom <= window.innerHeight && 
                   rect.right <= window.innerWidth;
        },

        // Focus sur le premier élément focusable
        focusFirstFocusable: function(container) {
            const focusableElements = container.querySelectorAll(
                'button, [href], input, select, textarea, [tabindex]:not([tabindex="-1"])'
            );
            
            if (focusableElements.length > 0) {
                focusableElements[0].focus();
            }
        }
    };

})(); 